<?php
/**
 * Módulo Database - Migraciones puntuales de esquema
 * Ejecuta ajustes para alinear columnas con el modelo actual
 */

$current_module = 'database';

if (!isLoggedIn()) {
    header('Location: ?module=auth&action=login');
    exit;
}

$db = getDB();
$pdo = $db->getConnection();

function columnExists(PDO $pdo, $table, $column) {
    // Algunos motores no aceptan parámetros en SHOW COLUMNS; usamos quoting manual
    $columnQuoted = $pdo->quote($column);
    $stmt = $pdo->query("SHOW COLUMNS FROM `$table` LIKE $columnQuoted");
    return (bool)$stmt->fetch();
}

$messages = [];
$errors = [];

try {
    // Asegurar columna job_title (alias de 'position' en UI)
    if (!columnExists($pdo, 'leads', 'job_title')) {
        $pdo->exec("ALTER TABLE `leads` ADD COLUMN `job_title` VARCHAR(100) NULL AFTER `company`");
        $messages[] = 'Columna job_title creada en leads.';
    } else {
        $messages[] = 'Columna job_title ya existe.';
    }

    // Si existe columna legacy 'position', copiar valores a job_title
    if (columnExists($pdo, 'leads', 'position')) {
        $pdo->exec("UPDATE `leads` SET `job_title` = COALESCE(`job_title`, `position`) WHERE `position` IS NOT NULL");
        $messages[] = 'Valores de position copiados a job_title.';
    }

    // Asegurar columnas de presencia y última vez online
    if (!columnExists($pdo, 'leads', 'is_online')) {
        $pdo->exec("ALTER TABLE `leads` ADD COLUMN `is_online` TINYINT(1) DEFAULT 0 AFTER `updated_at`");
        $messages[] = 'Columna is_online creada en leads.';
    } else {
        $messages[] = 'Columna is_online ya existe.';
    }

    if (!columnExists($pdo, 'leads', 'last_seen')) {
        $pdo->exec("ALTER TABLE `leads` ADD COLUMN `last_seen` DATETIME NULL AFTER `is_online`");
        $messages[] = 'Columna last_seen creada en leads.';
    } else {
        $messages[] = 'Columna last_seen ya existe.';
    }

    // Asegurar columnas para cachear última nota
    if (!columnExists($pdo, 'leads', 'last_note')) {
        $pdo->exec("ALTER TABLE `leads` ADD COLUMN `last_note` TEXT NULL AFTER `notes`");
        $messages[] = 'Columna last_note creada en leads.';
    } else {
        $messages[] = 'Columna last_note ya existe.';
    }

    if (!columnExists($pdo, 'leads', 'last_note_at')) {
        $pdo->exec("ALTER TABLE `leads` ADD COLUMN `last_note_at` DATETIME NULL AFTER `last_note`");
        $messages[] = 'Columna last_note_at creada en leads.';
    } else {
        $messages[] = 'Columna last_note_at ya existe.';
    }

    // Inicializar cache de última nota desde lead_activities (nota más reciente por lead)
    $pdo->exec(
        "UPDATE leads l
         LEFT JOIN (
           SELECT la.lead_id, la.description, la.created_at
           FROM lead_activities la
           INNER JOIN (
             SELECT lead_id, MAX(created_at) AS max_created
             FROM lead_activities
             WHERE type = 'note'
             GROUP BY lead_id
           ) m ON m.lead_id = la.lead_id AND m.max_created = la.created_at
         ) latest ON latest.lead_id = l.id
         SET l.last_note = latest.description, l.last_note_at = latest.created_at"
    );
    $messages[] = 'Cache de última nota inicializada.';

} catch (Exception $e) {
    $errors[] = $e->getMessage();
}

// Ejecutar datos de ejemplo si se solicita
if (isset($_GET['seed']) && $_GET['seed'] === '1') {
    try {
        $seedFile = BASE_PATH . '/database/seed_sample_data.sql';
        if (!file_exists($seedFile)) {
            $errors[] = 'Archivo de seed no encontrado.';
        } else {
            $sql = file_get_contents($seedFile);
            $statements = array_filter(array_map('trim', explode(';', $sql)), function($s) {
                return $s !== '' && strpos($s, '--') !== 0;
            });
            $pdo->beginTransaction();
            foreach ($statements as $stmt) {
                $pdo->exec($stmt);
            }
            $pdo->commit();
            $messages[] = 'Seeds de ejemplo cargados correctamente.';
        }
    } catch (Exception $e) {
        if ($pdo->inTransaction()) { $pdo->rollBack(); }
        $errors[] = 'Error al ejecutar seeds: ' . $e->getMessage();
    }
}

// Render simple
$page_title = 'Migraciones BD';
$module = 'database';
$action = 'migrate';
$content_file = null;
include BASE_PATH . '/modules/shared/layout.php';
?>
<div class="card" style="margin:1rem;">
  <div class="card-header"><h3>Resultado de migración</h3></div>
  <div class="card-body">
    <?php if (!empty($errors)): ?>
      <div class="alert alert-error">Error: <?php echo htmlspecialchars(implode('; ', $errors)); ?></div>
    <?php else: ?>
      <div class="alert alert-success">Migración ejecutada correctamente.</div>
      <ul>
        <?php foreach ($messages as $m): ?>
          <li><?php echo htmlspecialchars($m); ?></li>
        <?php endforeach; ?>
      </ul>
    <?php endif; ?>
    <div class="d-flex" style="gap:.5rem;">
      <a href="?module=database&action=migrate&seed=1" class="btn btn-secondary">Cargar datos de prueba</a>
      <a href="?module=leads" class="btn btn-primary">Ir a Leads</a>
    </div>
  </div>
</div>