<?php
/**
 * Autoloader para las clases API
 * Carga automáticamente las clases de integración cuando se necesiten
 */

class ApiLoader {
    private static $loaded = false;
    
    /**
     * Registra el autoloader
     */
    public static function register() {
        if (self::$loaded) {
            return;
        }
        
        spl_autoload_register([__CLASS__, 'loadClass']);
        self::$loaded = true;
    }
    
    /**
     * Carga una clase automáticamente
     */
    public static function loadClass($className) {
        $apiPath = dirname(__DIR__);
        
        // Mapeo de clases a archivos
        $classMap = [
            'Big4payClient' => $apiPath . '/big4pay/Big4payClient.php',
            'VoisoClient' => $apiPath . '/voiso/VoisoClient.php'
        ];
        
        if (isset($classMap[$className])) {
            $file = $classMap[$className];
            if (file_exists($file)) {
                require_once $file;
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Obtiene una instancia del cliente Big4pay
     */
    public static function getBig4payClient($pdo = null) {
        self::register();
        return new Big4payClient($pdo);
    }
    
    /**
     * Obtiene una instancia del cliente Voiso
     */
    public static function getVoisoClient($pdo = null) {
        self::register();
        return new VoisoClient($pdo);
    }
    
    /**
     * Verifica si una integración está disponible
     */
    public static function isIntegrationAvailable($integration) {
        $integrations = ['big4pay', 'voiso'];
        return in_array(strtolower($integration), $integrations);
    }
    
    /**
     * Obtiene todas las integraciones disponibles
     */
    public static function getAvailableIntegrations() {
        return [
            'big4pay' => [
                'name' => 'Big4pay',
                'description' => 'Procesador de pagos',
                'class' => 'Big4payClient'
            ],
            'voiso' => [
                'name' => 'Voiso',
                'description' => 'Sistema de llamadas',
                'class' => 'VoisoClient'
            ]
        ];
    }
}