<?php
/**
 * Script para corregir TODAS las rutas AJAX absolutas incorrectas
 * Problema: fetch('/modules/...') no funciona en producción
 * Solución: Convertir a rutas relativas apropiadas
 */

echo "<h1>🔧 Corrección Masiva de Rutas AJAX</h1>";
echo "<p>Corrigiendo rutas absolutas incorrectas en todo el sistema...</p>";
echo "<hr>";

// Lista de archivos con rutas problemáticas identificadas
$files_to_fix = [
    // Módulo Desks
    'modules/desks/document_review.php' => [
        "fetch('/modules/desks/update_document_status.php'" => "fetch('update_document_status.php'"
    ],
    'modules/desks/ticket.php' => [
        "fetch('/modules/desks/reply_ticket_message.php'" => "fetch('reply_ticket_message.php'"
    ],
    
    // Módulo Client Portal
    'modules/client_portal/ticket.php' => [
        "fetch('/modules/client_portal/add_ticket_message.php'" => "fetch('add_ticket_message.php'"
    ],
    'modules/client_portal/tickets.php' => [
        "fetch('/modules/client_portal/create_ticket.php'" => "fetch('create_ticket.php'"
    ],
    
    // Módulo Clients
    'modules/clients/view.php' => [
        "fetch('/modules/integrations/actions.php?action=voiso_click_to_call'" => "fetch('../integrations/actions.php?action=voiso_click_to_call'"
    ],
    
    // Módulo Integrations
    'modules/integrations/integrations_content.php' => [
        "fetch('/modules/integrations/actions.php?action=save_settings'" => "fetch('actions.php?action=save_settings'",
        "fetch('/modules/integrations/actions.php?action=test_voiso'" => "fetch('actions.php?action=test_voiso'",
        "fetch('/modules/integrations/actions.php?action=test_big4pay'" => "fetch('actions.php?action=test_big4pay'",
        "fetch('/modules/integrations/actions.php?action=get_big4pay_password'" => "fetch('actions.php?action=get_big4pay_password'",
        "fetch('/modules/integrations/actions.php?action=test_big4pay_payment'" => "fetch('actions.php?action=test_big4pay_payment'"
    ],
    
    // Dashboard público
    'public/dashboard.js' => [
        "fetch('/modules/client_portal/upload_document.php'" => "fetch('../modules/client_portal/upload_document.php'",
        "fetch('/modules/client_portal/create_ticket.php'" => "fetch('../modules/client_portal/create_ticket.php'"
    ]
];

$total_corrections = 0;
$total_files = 0;

foreach ($files_to_fix as $file_path => $replacements) {
    $full_path = __DIR__ . '/' . $file_path;
    
    if (!file_exists($full_path)) {
        echo "⚠️ Archivo no encontrado: $file_path<br>";
        continue;
    }
    
    $content = file_get_contents($full_path);
    $original_content = $content;
    $file_corrections = 0;
    
    foreach ($replacements as $search => $replace) {
        $count = 0;
        $content = str_replace($search, $replace, $content, $count);
        if ($count > 0) {
            echo "✓ $file_path: '$search' → '$replace' ($count veces)<br>";
            $file_corrections += $count;
            $total_corrections += $count;
        }
    }
    
    if ($content !== $original_content) {
        if (file_put_contents($full_path, $content)) {
            echo "✅ <strong>$file_path</strong> - $file_corrections correcciones aplicadas<br>";
            $total_files++;
        } else {
            echo "❌ Error al escribir: $file_path<br>";
        }
    } else {
        echo "ℹ️ Sin cambios necesarios: $file_path<br>";
    }
    
    echo "<br>";
}

echo "<hr>";
echo "<h2>📊 Resumen de Correcciones</h2>";
echo "<ul>";
echo "<li><strong>Archivos procesados:</strong> " . count($files_to_fix) . "</li>";
echo "<li><strong>Archivos modificados:</strong> $total_files</li>";
echo "<li><strong>Total de correcciones:</strong> $total_corrections</li>";
echo "</ul>";

echo "<h3>🎯 Tipos de correcciones aplicadas:</h3>";
echo "<ol>";
echo "<li><strong>Rutas del mismo módulo:</strong> <code>/modules/modulo/archivo.php</code> → <code>archivo.php</code></li>";
echo "<li><strong>Rutas entre módulos:</strong> <code>/modules/otro/archivo.php</code> → <code>../otro/archivo.php</code></li>";
echo "<li><strong>Rutas desde public:</strong> <code>/modules/modulo/archivo.php</code> → <code>../modules/modulo/archivo.php</code></li>";
echo "</ol>";

echo "<h3>✅ Beneficios:</h3>";
echo "<ul>";
echo "<li>✅ Funciona en desarrollo (XAMPP)</li>";
echo "<li>✅ Funciona en producción (public_html/test/simple_crm)</li>";
echo "<li>✅ No más errores JSON \"Unexpected token '<'\"</li>";
echo "<li>✅ Rutas relativas son más portables</li>";
echo "</ul>";

echo "<h3>🧪 Próximos pasos:</h3>";
echo "<ol>";
echo "<li>Sube todos los archivos corregidos al servidor</li>";
echo "<li>Prueba cada módulo para verificar que las funciones AJAX funcionen</li>";
echo "<li>Revisa los logs de errores para confirmar que no hay más problemas</li>";
echo "</ol>";

echo "<p><a href='test_roles_ajax.php'>🧪 Probar Roles AJAX</a> | <a href='index.php'>🏠 Ir al Sistema</a></p>";

// Crear archivo de log con los cambios
$log_content = "=== CORRECCIONES AJAX APLICADAS ===\n";
$log_content .= "Fecha: " . date('Y-m-d H:i:s') . "\n";
$log_content .= "Archivos modificados: $total_files\n";
$log_content .= "Total correcciones: $total_corrections\n\n";

foreach ($files_to_fix as $file_path => $replacements) {
    $log_content .= "Archivo: $file_path\n";
    foreach ($replacements as $search => $replace) {
        $log_content .= "  - '$search' → '$replace'\n";
    }
    $log_content .= "\n";
}

file_put_contents(__DIR__ . '/logs/ajax_corrections_' . date('Y-m-d_H-i-s') . '.log', $log_content);
echo "<p>📝 Log guardado en: logs/ajax_corrections_" . date('Y-m-d_H-i-s') . ".log</p>";
?>