<?php
/**
 * Endpoint para recibir logs de errores del frontend
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Solo permitir POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Obtener datos del error
    $input = file_get_contents('php://input');
    $errorData = json_decode($input, true);
    
    if (!$errorData) {
        throw new Exception('Invalid JSON data');
    }
    
    // Agregar información del servidor
    $logEntry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'server_time' => time(),
        'client_ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
        'referer' => $_SERVER['HTTP_REFERER'] ?? 'unknown',
        'error_data' => $errorData
    ];
    
    // Crear directorio de logs si no existe
    $logDir = __DIR__ . '/logs';
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    // Nombre del archivo de log
    $logFile = $logDir . '/frontend_errors_' . date('Y-m-d') . '.log';
    
    // Escribir al archivo de log
    $logLine = date('Y-m-d H:i:s') . ' | ' . json_encode($logEntry) . PHP_EOL;
    file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
    
    // También escribir a un log general
    $generalLogFile = $logDir . '/all_errors.log';
    file_put_contents($generalLogFile, $logLine, FILE_APPEND | LOCK_EX);
    
    // Log crítico para errores de conexión
    if (isset($errorData['type']) && 
        (strpos($errorData['type'], 'Network') !== false || 
         strpos($errorData['type'], 'HTTP') !== false ||
         strpos($errorData['message'], 'conexión') !== false)) {
        
        $criticalLogFile = $logDir . '/critical_errors.log';
        $criticalEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'type' => 'CRITICAL_CONNECTION_ERROR',
            'error' => $errorData,
            'server_info' => [
                'php_version' => PHP_VERSION,
                'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'unknown',
                'document_root' => $_SERVER['DOCUMENT_ROOT'] ?? 'unknown'
            ]
        ];
        
        $criticalLine = date('Y-m-d H:i:s') . ' | CRITICAL | ' . json_encode($criticalEntry) . PHP_EOL;
        file_put_contents($criticalLogFile, $criticalLine, FILE_APPEND | LOCK_EX);
    }
    
    // Respuesta exitosa
    echo json_encode([
        'success' => true,
        'message' => 'Error logged successfully',
        'log_file' => basename($logFile)
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Failed to log error: ' . $e->getMessage()
    ]);
    
    // Log del error del sistema de logging
    error_log('Error Logger Failed: ' . $e->getMessage());
}
?>