<?php
session_start();
if (!isset($_SESSION['client_id'])) { header('Location: login.php'); exit(); }
if (!defined('BASE_PATH')) { define('BASE_PATH', dirname(dirname(__DIR__))); }
require_once BASE_PATH . '/database/config.php';
require_once BASE_PATH . '/database/connection.php';
require_once BASE_PATH . '/api/shared/ApiLoader.php'; // Incluir el autoloader de API

$page_title = 'Depósito';
$module = 'transactions';
$action = 'deposit';

$clientId = (int)$_SESSION['client_id'];
$db = Database::getInstance();
$pdo = $db->getConnection();

// Usar la nueva clase API para verificar configuración de Big4pay
$big4payClient = ApiLoader::getBig4payClient($pdo);
$big4payEnabled = $big4payClient->isConfigured();

$stmtAcc = $pdo->prepare("SELECT id, account_number, currency, account_type, balance FROM client_accounts WHERE client_id = ? AND status = 'active' ORDER BY created_at ASC");
$stmtAcc->execute([$clientId]);
$accounts = $stmtAcc->fetchAll();

$message = null; $error = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accountId = (int)($_POST['account_id'] ?? 0);
    $amount = (float)($_POST['amount'] ?? 0);
    $method = trim($_POST['method'] ?? 'bank');

    if ($accountId <= 0 || $amount <= 0) {
        $error = 'Selecciona una cuenta y un monto válido.';
    } else {
        // Si el método es Big4pay, iniciar pago real con datos del cliente
        if ($method === 'big4pay' && $big4payEnabled) {
            // Obtener cuenta seleccionada (para moneda)
            $stmtOne = $pdo->prepare("SELECT id, account_number, currency FROM client_accounts WHERE id = ? AND client_id = ? LIMIT 1");
            $stmtOne->execute([$accountId, $clientId]);
            $acc = $stmtOne->fetch();
            if (!$acc) { $error = 'Cuenta no encontrada.'; }
            else {
                $currency = $acc['currency'] ?? 'USD';
                // Obtener datos del cliente para el objeto customer
                $stmtCli = $pdo->prepare("SELECT first_name, last_name, email, phone FROM clients WHERE id = ? LIMIT 1");
                $stmtCli->execute([$clientId]);
                $clientRow = $stmtCli->fetch();
                $firstName = $clientRow['first_name'] ?? 'Cliente';
                $lastName = $clientRow['last_name'] ?? '';
                $email = $clientRow['email'] ?? '';
                $mobile = $clientRow['phone'] ?? '';

                // Paso 1: crear token y guardar Authorization
                $raw = $big4payClient->createAccessTokenRaw();
                if (!$raw['success']) {
                    $error = 'Error autenticando con Big4pay: ' . ($raw['message'] ?? ('HTTP ' . ($raw['http_code'] ?? '')));
                } else {
                    $token = $raw['authorization'] ?? ($raw['json']['access_token'] ?? null);
                    if ($token) { $big4payClient->saveAccessToken($token); }

                    // Paso 2: identifier incremental TRX_XXXXXXXX
                    $identifier = $big4payClient->generateIdentifier();

                    // Paso 3: URLs configuradas o absolutas
                    $config = $big4payClient->getConfiguration();
                    $ipnUrl = $config['ipn_url'] ?? null;
                    $successUrl = $config['success_url'] ?? null;
                    $cancelUrl = $config['cancel_url'] ?? null;
                    if (!$ipnUrl || !$successUrl || !$cancelUrl) {
                        $scheme = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
                        $origin = $scheme . '://' . ($_SERVER['HTTP_HOST'] ?? 'localhost');
                        $rootPath = rtrim(dirname(dirname(dirname($_SERVER['SCRIPT_NAME']))), '/');
                        $ipnUrl = $ipnUrl ?: ($origin . $rootPath . '/api/big4pay/ipn.php');
                        $successUrl = $successUrl ?: ($origin . $rootPath . '/modules/client_portal/big4pay_success.php');
                        $cancelUrl = $cancelUrl ?: ($origin . $rootPath . '/modules/client_portal/big4pay_cancel.php');
                    }

                    // Paso 4: construir payload e iniciar pago en endpoint Live
                    $params = [
                        'access_token' => $token,
                        'identifier' => $identifier,
                        'currency' => $currency,
                        'amount' => $amount,
                        'language' => 'ES',
                        'description' => 'Depósito de cliente',
                        'details' => 'deposit transactions',
                        'ipn_url' => $ipnUrl,
                        'success_url' => $successUrl,
                        'cancel_url' => $cancelUrl,
                        'site_name' => 'IAttrade',
                        'site_logo' => 'http://yoursite.com/logo.png',
                        'checkout_theme' => 'light',
                        'customer' => [
                            'first_name' => $firstName,
                            'last_name' => $lastName,
                            'country' => ($currency === 'MXN' ? 'MX' : 'US'),
                            'email' => $email,
                            'mobile' => $mobile
                        ]
                    ];
                    $res = $big4payClient->initiatePayment($params);
                    // Guardar datos del depósito en sesión por trazabilidad
                    $_SESSION['pending_deposit'] = [
                        'account_id' => $accountId,
                        'amount' => $amount,
                        'identifier' => $identifier,
                        'access_token' => $token
                    ];

                    // Determinar URL de redirección a checkout
                    $redirectUrl = null;
                    if (is_array($res)) {
                        $json = $res['json'] ?? null;
                        if (is_array($json)) {
                            $redirectUrl = $json['redirect_url'] ?? $json['checkout_url'] ?? $json['payment_link'] ?? null;
                        }
                        if (!$redirectUrl && isset($res['body']) && is_string($res['body'])) {
                            $decoded = json_decode($res['body'], true);
                            if (is_array($decoded)) {
                                $redirectUrl = $decoded['redirect_url'] ?? $decoded['checkout_url'] ?? $decoded['payment_link'] ?? null;
                            }
                        }
                    }

                    if ($redirectUrl) {
                        header('Location: ' . $redirectUrl);
                        exit;
                    } else {
                        $error = 'No se recibió URL de checkout. ' . (isset($res['body']) ? (is_string($res['body']) ? $res['body'] : json_encode($res['body'])) : '');
                    }
                }
            }
        } else {
            // Flujo normal para otros métodos de pago
            $stmtOne = $pdo->prepare("SELECT id, currency, balance FROM client_accounts WHERE id = ? AND client_id = ? LIMIT 1");
            $stmtOne->execute([$accountId, $clientId]);
            $acc = $stmtOne->fetch();
            if (!$acc) { $error = 'Cuenta no encontrada.'; }
            else {
                $currency = $acc['currency'] ?? 'USD';
                $pdo->beginTransaction();
                try {
                    $stmtTx = $pdo->prepare("INSERT INTO client_transactions (client_id, type, amount, currency, reference, notes, created_at) VALUES (?,?,?,?,?, ?, NOW())");
                    $stmtTx->execute([$clientId, 'deposit', $amount, $currency, uniqid('DEP-'), $method]);

                    $stmtUp = $pdo->prepare("UPDATE client_accounts SET balance = balance + ? WHERE id = ?");
                    $stmtUp->execute([$amount, $accountId]);

                    $pdo->commit();
                    $message = 'Depósito registrado correctamente.';
                } catch (Throwable $e) {
                    $pdo->rollBack();
                    $error = 'Error registrando el depósito.';
                }
            }
        }
    }
}

ob_start();
?>
    <div class="section-header">
        <h2 class="section-title"><i class="fas fa-arrow-down"></i> Nuevo Depósito</h2>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
    <?php elseif ($error): ?>
        <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <section class="support-card">
        <form method="post" class="ticket-form">
            <div class="form-grid">
                <div class="form-group">
                    <label for="account_id">Cuenta</label>
                    <select id="account_id" name="account_id" required>
                        <option value="">Selecciona una cuenta</option>
                        <?php foreach ($accounts as $a): ?>
                            <option value="<?php echo (int)$a['id']; ?>" <?php echo ((int)($_GET['account_id'] ?? 0) === (int)$a['id']) ? 'selected' : ''; ?>>
                                <?php echo strtoupper($a['currency']) . ' · #' . htmlspecialchars($a['account_number']) . ' · ' . ucfirst($a['account_type']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label for="amount">Monto</label>
                    <input type="number" id="amount" name="amount" step="0.01" min="0.01" required />
                </div>
                <div class="form-group">
                    <label for="method">Método</label>
                    <select id="method" name="method" required>
                        <option value="bank">Transferencia bancaria</option>
                        <option value="card">Tarjeta</option>
                        <option value="crypto">Cripto</option>
                        <?php if ($big4payEnabled): ?>
                            <option value="big4pay">Big4pay</option>
                        <?php endif; ?>
                    </select>
                </div>
            </div>
            <div class="form-actions">
                <button class="client-action-btn deposit" type="submit"><i class="fas fa-arrow-down"></i> Depositar</button>
                <a class="client-action-btn secondary" href="/modules/client_portal/dashboard.php">Cancelar</a>
            </div>
        </form>
    </section>
<?php
$content = ob_get_clean();
include __DIR__ . '/layout.php';
?>