<?php
// Módulo Clientes - Listado
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (!defined('BASE_PATH')) { define('BASE_PATH', dirname(dirname(__DIR__))); }
require_once BASE_PATH . '/database/config.php';
require_once BASE_PATH . '/database/connection.php';

if (!function_exists('isLoggedIn')) {
    function isLoggedIn(){ return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']); }
}
if (!isLoggedIn()) { header('Location: /modules/auth/login.php'); exit; }

$db = Database::getInstance();
$pdo = $db->getConnection();

// Helper de permisos por nombre
if (!function_exists('userHasPermissionByName')) {
    function userHasPermissionByName(PDO $pdo, string $permName, int $userId): bool {
        if (empty($userId)) return false;
        $stmt = $pdo->prepare("SELECT 1 FROM role_permissions rp INNER JOIN permissions p ON rp.permission_id = p.id INNER JOIN user_roles ur ON rp.role_id = ur.role_id WHERE ur.user_id = ? AND p.name = ? LIMIT 1");
        $stmt->execute([$userId, $permName]);
        return (bool)$stmt->fetchColumn();
    }
}

// Solo Administrador puede acceder al módulo de Clientes
if (!function_exists('isAdminRole')) {
    function isAdminRole(PDO $pdo, int $userId): bool {
        if ($userId <= 0) return false;
        $stmt = $pdo->prepare("SELECT 1 FROM user_roles ur INNER JOIN roles r ON ur.role_id = r.id WHERE ur.user_id = ? AND r.name = 'admin' LIMIT 1");
        $stmt->execute([$userId]);
        return (bool)$stmt->fetchColumn();
    }
}
$current_user_id = (int)($_SESSION['user_id'] ?? 0);
if (!isAdminRole($pdo, $current_user_id)) { http_response_code(403); echo 'Acceso restringido a Administrador'; exit; }

// Helpers de enmascarado
if (!function_exists('maskGeneric')) {
    function maskGeneric($value, $first = 3, $last = 2) {
        $v = (string)$value; $len = strlen($v);
        if ($len <= ($first + $last)) { return str_repeat('*', max(0, $len)); }
        return substr($v, 0, $first) . str_repeat('*', $len - ($first + $last)) . substr($v, -$last);
    }
}
if (!function_exists('maskEmail')) {
    function maskEmail($email, $first = 3, $last = 2) {
        $e = (string)$email;
        if (strpos($e, '@') === false) { return maskGeneric($e, $first, $last); }
        list($local, $domain) = explode('@', $e, 2);
        $localMasked = maskGeneric($local, $first, $last);
        $domainLen = strlen($domain);
        $domainMasked = $domainLen <= $last ? str_repeat('*', max(0, $domainLen)) : str_repeat('*', $domainLen - $last) . substr($domain, -$last);
        return $localMasked . '@' . $domainMasked;
    }
}
if (!function_exists('maskPhone')) {
    function maskPhone($phone, $first = 3, $last = 2) {
        $digits = preg_replace('/\D+/', '', (string)$phone);
        $len = strlen($digits);
        if ($len === 0) { return ''; }
        $visibleStart = min($first, max(0, $len));
        $visibleEnd = min($last, max(0, $len - $visibleStart));
        return substr($digits, 0, $visibleStart) . str_repeat('*', max(0, $len - ($visibleStart + $visibleEnd))) . substr($digits, -$visibleEnd);
    }
}

// Permisos de visibilidad completa en Clientes
$canViewClientEmailFull = userHasPermissionByName($pdo, 'client_email_mask', $current_user_id)
    || userHasPermissionByName($pdo, 'clients.view_email_full', $current_user_id);
$canViewClientPhoneFull = userHasPermissionByName($pdo, 'client_number_mask', $current_user_id)
    || userHasPermissionByName($pdo, 'clients.view_phone_full', $current_user_id);

// Obtener clientes básicos
$stmt = $pdo->query("SELECT 
                         c.id, c.first_name, c.last_name, c.email, c.phone, c.status, c.created_at, 
                         c.balance, c.equity, c.margin_percent, c.pnl,
                         l.id AS lead_id,
                         d.name AS desk_name
                     FROM clients c 
                     LEFT JOIN leads l ON c.lead_id = l.id
                     LEFT JOIN desks d ON c.desk_id = d.id
                     ORDER BY c.created_at DESC");
$clients = $stmt->fetchAll();
// Integración con layout central
$page_title = 'Clientes';
$current_module = 'clients';
$current_action = 'index';

// Widgets de estadísticas de clientes
$totalClients = $activeClients = $inactiveClients = $suspendedClients = 0;
try {
    $totalClients = (int)$pdo->query("SELECT COUNT(*) FROM clients")->fetchColumn();
    $activeClients = (int)$pdo->query("SELECT COUNT(*) FROM clients WHERE status = 'active'")->fetchColumn();
    $inactiveClients = (int)$pdo->query("SELECT COUNT(*) FROM clients WHERE status = 'inactive'")->fetchColumn();
    $suspendedClients = (int)$pdo->query("SELECT COUNT(*) FROM clients WHERE status = 'suspended'")->fetchColumn();
} catch (Exception $e) { /* evitar romper vista si falta tabla */ }

ob_start();
?>
<div class="roles-header">
    <div class="header-left">
        <h1 class="page-title">
            <svg class="title-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M16 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                <circle cx="12" cy="7" r="4"/>
            </svg>
            Gestión de Clientes
        </h1>
        <p class="page-subtitle">Administra clientes, estados y actividad financiera.</p>
    </div>
</div>
<section class="stats-grid" style="margin-bottom:1rem;">
    <div class="stat-card" data-stat-key="total">
        <div class="stat-icon bg-blue-500">
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M16 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                <circle cx="12" cy="7" r="4"/>
            </svg>
        </div>
        <div class="stat-content">
            <div class="stat-number"><?= number_format($totalClients, 0, ',', '.') ?></div>
            <div class="stat-label">Total</div>
        </div>
    </div>
    <div class="stat-card" data-stat-key="active">
        <div class="stat-icon bg-green-500">
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M9 12l2 2 4-4"/>
                <circle cx="12" cy="12" r="10"/>
            </svg>
        </div>
        <div class="stat-content">
            <div class="stat-number"><?= number_format($activeClients, 0, ',', '.') ?></div>
            <div class="stat-label">Activos</div>
        </div>
    </div>
    <div class="stat-card" data-stat-key="inactive">
        <div class="stat-icon bg-yellow-500">
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="9"/>
                <path d="M9 12h6"/>
            </svg>
        </div>
        <div class="stat-content">
            <div class="stat-number"><?= number_format($inactiveClients, 0, ',', '.') ?></div>
            <div class="stat-label">Inactivos</div>
        </div>
    </div>
    <div class="stat-card" data-stat-key="suspended">
        <div class="stat-icon bg-teal-500">
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="9"/>
                <path d="M4 4l16 16"/>
            </svg>
        </div>
        <div class="stat-content">
            <div class="stat-number"><?= number_format($suspendedClients, 0, ',', '.') ?></div>
            <div class="stat-label">Suspendidos</div>
        </div>
    </div>
</section>
<section class="card">
    <div class="card-body">
        <table class="table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Nombre</th>
                    <th>Email</th>
                    <th>Teléfono</th>
                    <th>Desk</th>
                    <th>Estado</th>
                    <th>Balance</th>
                    <th>Equity</th>
                    <th>% Margen</th>
                    <th>PnL</th>
                    <th>Creado</th>
                    <th>Acciones</th>
                </tr>
            </thead>
            <tbody>
            <?php foreach ($clients as $c): ?>
                <tr>
                    <td><?= htmlspecialchars($c['id']) ?></td>
                    <td><?= htmlspecialchars($c['first_name'] . ' ' . $c['last_name']) ?></td>
                    <?php 
                        $emailDisplay = $canViewClientEmailFull ? ($c['email'] ?? '') : maskEmail($c['email'] ?? '');
                        $phoneDisplay = $canViewClientPhoneFull ? ($c['phone'] ?? '') : maskPhone($c['phone'] ?? '');
                    ?>
                    <td><?= htmlspecialchars($emailDisplay) ?></td>
                    <td><?= htmlspecialchars($phoneDisplay) ?></td>
                    <td><?= htmlspecialchars($c['desk_name'] ?? '') ?></td>
                    <td><span class="status-badge bg-teal-100 text-teal-800"><?= htmlspecialchars($c['status']) ?></span></td>
                    <td><?= number_format((float)($c['balance'] ?? 0), 2, ',', '.') ?></td>
                    <td><?= number_format((float)($c['equity'] ?? 0), 2, ',', '.') ?></td>
                    <td><?= number_format((float)($c['margin_percent'] ?? 0), 2, ',', '.') ?>%</td>
                    <td><?= number_format((float)($c['pnl'] ?? 0), 2, ',', '.') ?></td>
                    <td><?= htmlspecialchars($c['created_at']) ?></td>
                    <td>
                        <a href="/modules/clients/view.php?id=<?= intval($c['id']) ?>" class="btn btn-secondary btn-size-md">Ver</a>
                    </td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</section>
<?php
$content = ob_get_clean();
include BASE_PATH . '/modules/shared/layout.php';
return;