<?php
// Detalle de Cliente
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (!defined('BASE_PATH')) { define('BASE_PATH', dirname(dirname(__DIR__))); }
require_once BASE_PATH . '/database/config.php';
require_once BASE_PATH . '/database/connection.php';
if (!function_exists('isLoggedIn')) {
    function isLoggedIn(){ return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']); }
}
if (!isLoggedIn()) { header('Location: /modules/auth/login.php'); exit; }

$id = isset($_GET['id']) ? intval($_GET['id']) : 0;
if ($id <= 0) { http_response_code(400); echo 'Cliente inválido'; exit; }

$db = Database::getInstance();
$pdo = $db->getConnection();

// Helper de permisos por nombre
if (!function_exists('userHasPermissionByName')) {
    function userHasPermissionByName(PDO $pdo, string $permName, int $userId): bool {
        if (empty($userId)) return false;
        $stmt = $pdo->prepare("SELECT 1 FROM role_permissions rp INNER JOIN permissions p ON rp.permission_id = p.id INNER JOIN user_roles ur ON rp.role_id = ur.role_id WHERE ur.user_id = ? AND p.name = ? LIMIT 1");
        $stmt->execute([$userId, $permName]);
        return (bool)$stmt->fetchColumn();
    }
}

// Solo Administrador puede acceder al detalle de Clientes
if (!function_exists('isAdminRole')) {
    function isAdminRole(PDO $pdo, int $userId): bool {
        if ($userId <= 0) return false;
        $stmt = $pdo->prepare("SELECT 1 FROM user_roles ur INNER JOIN roles r ON ur.role_id = r.id WHERE ur.user_id = ? AND r.name = 'admin' LIMIT 1");
        $stmt->execute([$userId]);
        return (bool)$stmt->fetchColumn();
    }
}
$current_user_id = (int)($_SESSION['user_id'] ?? 0);
if (!isAdminRole($pdo, $current_user_id)) { http_response_code(403); echo 'Acceso restringido a Administrador'; exit; }

// Helpers de enmascarado
if (!function_exists('maskGeneric')) {
    function maskGeneric($value, $first = 3, $last = 2) {
        $v = (string)$value; $len = strlen($v);
        if ($len <= ($first + $last)) { return str_repeat('*', max(0, $len)); }
        return substr($v, 0, $first) . str_repeat('*', $len - ($first + $last)) . substr($v, -$last);
    }
}
if (!function_exists('maskEmail')) {
    function maskEmail($email, $first = 3, $last = 2) {
        $e = (string)$email;
        if (strpos($e, '@') === false) { return maskGeneric($e, $first, $last); }
        list($local, $domain) = explode('@', $e, 2);
        $localMasked = maskGeneric($local, $first, $last);
        $domainLen = strlen($domain);
        $domainMasked = $domainLen <= $last ? str_repeat('*', max(0, $domainLen)) : str_repeat('*', $domainLen - $last) . substr($domain, -$last);
        return $localMasked . '@' . $domainMasked;
    }
}
if (!function_exists('maskPhone')) {
    function maskPhone($phone, $first = 3, $last = 2) {
        $digits = preg_replace('/\D+/', '', (string)$phone);
        $len = strlen($digits);
        if ($len === 0) { return ''; }
        $visibleStart = min($first, max(0, $len));
        $visibleEnd = min($last, max(0, $len - $visibleStart));
        return substr($digits, 0, $visibleStart) . str_repeat('*', max(0, $len - ($visibleStart + $visibleEnd))) . substr($digits, -$visibleEnd);
    }
}

// Permisos de visibilidad completa en Clientes
$canViewClientEmailFull = userHasPermissionByName($pdo, 'client_email_mask', $current_user_id)
    || userHasPermissionByName($pdo, 'clients.view_email_full', $current_user_id);
$canViewClientPhoneFull = userHasPermissionByName($pdo, 'client_number_mask', $current_user_id)
    || userHasPermissionByName($pdo, 'clients.view_phone_full', $current_user_id);
// Permiso para Click-to-call (Voiso)
$canClickToCall = userHasPermissionByName($pdo, 'voiso.click_to_call', $current_user_id);

// Cliente
$stmt = $pdo->prepare("SELECT c.*, l.status AS lead_status FROM clients c LEFT JOIN leads l ON c.lead_id = l.id WHERE c.id = ?");
$stmt->execute([$id]);
$client = $stmt->fetch();
if (!$client) { http_response_code(404); echo 'Cliente no encontrado'; exit; }

// Cuentas
$stmtAcc = $pdo->prepare("SELECT * FROM client_accounts WHERE client_id = ? ORDER BY created_at DESC");
$stmtAcc->execute([$id]);
$accounts = $stmtAcc->fetchAll();

// Depósitos
$stmtDep = $pdo->prepare("SELECT * FROM client_deposits WHERE client_id = ? ORDER BY created_at DESC");
$stmtDep->execute([$id]);
$deposits = $stmtDep->fetchAll();

// Actividades del lead (historial) si existe
$activities = [];
if (!empty($client['lead_id'])) {
    $stmtAct = $pdo->prepare("SELECT * FROM lead_activities WHERE lead_id = ? ORDER BY created_at DESC");
    $stmtAct->execute([$client['lead_id']]);
    $activities = $stmtAct->fetchAll();
}

// Mensaje de clave provisional
$provisional = null;
if (isset($_SESSION['client_created_id']) && $_SESSION['client_created_id'] == $client['id'] && isset($_SESSION['client_provisional_password'])) {
    $provisional = $_SESSION['client_provisional_password'];
    unset($_SESSION['client_created_id']);
    unset($_SESSION['client_provisional_password']);
}
// Integración con layout central
$page_title = 'Cliente #' . (int)$client['id'];
$current_module = 'clients';
$current_action = 'view';
ob_start();
?>
<main class="container">
    <header class="page-header">
        <h1>Cliente: <?= htmlspecialchars($client['first_name'] . ' ' . $client['last_name']) ?></h1>
        <div class="header-actions">
            <a href="/modules/clients/index.php" class="btn btn-secondary btn-size-md">Volver</a>
        </div>
    </header>

    <?php if ($provisional): ?>
    <section class="alert alert-info">Clave provisional generada: <strong><?= htmlspecialchars($provisional) ?></strong> (cámbiala tras el primer acceso)</section>
    <?php endif; ?>

    <section class="card">
        <div class="card-body">
            <div class="grid grid-2">
                <div>
                    <h3>Perfil</h3>
                    <?php 
                        $emailDisplay = $canViewClientEmailFull ? ($client['email'] ?? '') : maskEmail($client['email'] ?? '');
                        $phoneDisplay = $canViewClientPhoneFull ? ($client['phone'] ?? '') : maskPhone($client['phone'] ?? '');
                    ?>
                    <p><strong>Email:</strong> <?= htmlspecialchars($emailDisplay) ?></p>
                    <p><strong>Teléfono:</strong> <?= htmlspecialchars($phoneDisplay) ?>
                    <?php if ($canClickToCall && !empty($client['phone'])): ?>
                        <button class="btn btn-secondary btn-size-sm" style="margin-left:8px" onclick="voisoClickToCall('client', <?= (int)$client['id'] ?>)">Llamar con Voiso</button>
                    <?php endif; ?></p>
                    <p><strong>Estado:</strong> <?= htmlspecialchars($client['status']) ?></p>
                    <p><strong>Lead Status:</strong> <?= htmlspecialchars($client['lead_status'] ?? 'N/A') ?></p>
                </div>
                <div>
                    <h3>Resumen</h3>
                    <p><strong>Cuentas:</strong> <?= count($accounts) ?></p>
                    <p><strong>Depósitos:</strong> <?= count($deposits) ?></p>
                </div>
            </div>
        </div>
    </section>

    <section class="card">
        <div class="card-body">
            <h3>Depósito Manual</h3>
            <form method="post" action="/modules/clients/manual_deposit.php" class="form-inline">
                <input type="hidden" name="client_id" value="<?= intval($client['id']) ?>">
                <input type="number" name="amount" step="0.01" placeholder="Monto" required class="input">
                <select name="currency" class="input">
                    <option value="USD">USD</option>
                    <option value="EUR">EUR</option>
                </select>
                <input type="text" name="method" placeholder="Método" value="manual" class="input">
                <input type="text" name="reference" placeholder="Referencia" class="input">
                <button type="submit" class="btn btn-primary btn-size-md">Registrar Depósito</button>
            </form>
        </div>
    </section>

    <section class="card">
        <div class="card-body">
            <h3>Cuentas de Trading</h3>
            <table class="table">
                <thead><tr><th>Tipo</th><th>Número</th><th>Divisa</th><th>Balance</th><th>Estado</th></tr></thead>
                <tbody>
                <?php foreach ($accounts as $a): ?>
                    <tr>
                        <td><?= htmlspecialchars($a['account_type']) ?></td>
                        <td><?= htmlspecialchars($a['account_number']) ?></td>
                        <td><?= htmlspecialchars($a['currency']) ?></td>
                        <td><?= htmlspecialchars($a['balance']) ?></td>
                        <td><?= htmlspecialchars($a['status']) ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </section>

    <section class="card">
        <div class="card-body">
            <h3>Depósitos</h3>
            <table class="table">
                <thead><tr><th>Monto</th><th>Divisa</th><th>Método</th><th>Referencia</th><th>Estado</th><th>Fecha</th></tr></thead>
                <tbody>
                <?php foreach ($deposits as $d): ?>
                    <tr>
                        <td><?= htmlspecialchars($d['amount']) ?></td>
                        <td><?= htmlspecialchars($d['currency']) ?></td>
                        <td><?= htmlspecialchars($d['method']) ?></td>
                        <td><?= htmlspecialchars($d['reference']) ?></td>
                        <td><?= htmlspecialchars($d['status']) ?></td>
                        <td><?= htmlspecialchars($d['created_at']) ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </section>

    <?php if (!empty($activities)): ?>
    <section class="card">
        <div class="card-body">
            <h3>Historial de Actividades (Lead)</h3>
            <ul class="activity-list">
                <?php foreach ($activities as $act): ?>
                    <li><strong><?= htmlspecialchars($act['type']) ?>:</strong> <?= htmlspecialchars($act['subject']) ?> <em>(<?= htmlspecialchars($act['created_at']) ?>)</em></li>
                <?php endforeach; ?>
            </ul>
        </div>
    </section>
    <?php endif; ?>
</main>
<script>
async function voisoClickToCall(entity, id) {
    const btns = document.querySelectorAll('.btn, .chip');
    btns.forEach(b => { if (b) b.disabled = true; });
    try {
        const resp = await fetch('../integrations/actions.php?action=voiso_click_to_call', {
            method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ entity, id })
        });
        const data = await resp.json();
        const msg = data.message || (data.success ? 'Llamada iniciada' : 'No se pudo iniciar la llamada');
        alert(msg);
    } catch (e) {
        alert('Error realizando Click-to-call');
    } finally {
        btns.forEach(b => { if (b) b.disabled = false; });
    }
}
</script>
<?php
$content = ob_get_clean();
include BASE_PATH . '/modules/shared/layout.php';
return;
?>