<?php
/**
 * Módulo Leads - Endpoint AJAX para agregar nota a un lead
 * Sistema de logging mejorado para diagnóstico de errores
 */

// Clean output buffer to prevent any unwanted output
ob_start();

// Include necessary files
require_once __DIR__ . '/../../database/config.php';
require_once __DIR__ . '/../../database/connection.php';
session_start();

// Enable error reporting for debugging but don't display errors to avoid JSON corruption
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Función de logging mejorada
function logError($message, $context = []) {
    $logDir = __DIR__ . '/../../logs';
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logFile = $logDir . '/add_note_errors_' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = [
        'timestamp' => $timestamp,
        'message' => $message,
        'context' => $context,
        'request_uri' => $_SERVER['REQUEST_URI'] ?? '',
        'method' => $_SERVER['REQUEST_METHOD'] ?? '',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'remote_addr' => $_SERVER['REMOTE_ADDR'] ?? ''
    ];
    
    $logLine = $timestamp . ' | ' . json_encode($logEntry) . PHP_EOL;
    file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
    
    // También log en error_log de PHP
    error_log("ADD_NOTE_ERROR: $message | Context: " . json_encode($context));
}

// Log del inicio de la request
logError('Request iniciada', [
    'method' => $_SERVER['REQUEST_METHOD'] ?? '',
    'content_type' => $_SERVER['CONTENT_TYPE'] ?? '',
    'content_length' => $_SERVER['CONTENT_LENGTH'] ?? '',
    'session_id' => session_id(),
    'user_id' => $_SESSION['user_id'] ?? 'no_session'
]);

// Define helper functions if not already defined
if (!function_exists('isLoggedIn')) {
    function isLoggedIn() {
        return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }
}

header('Content-Type: application/json; charset=utf-8');

// Clean any previous output
ob_clean();

try {
    if (!isLoggedIn()) {
        logError('Usuario no autorizado', ['session' => $_SESSION]);
        http_response_code(401);
        ob_end_clean(); // Limpiar buffer antes de enviar JSON
        echo json_encode(['success' => false, 'message' => 'No autorizado']);
        exit;
    }

    logError('Usuario autorizado', ['user_id' => $_SESSION['user_id']]);

    $db = getDB();
    
    // Test database connection
    if (!$db) {
        logError('Error de conexión a base de datos', ['db_config' => 'failed']);
        throw new Exception('No se pudo conectar a la base de datos');
    }
    
    logError('Conexión a base de datos exitosa');

// Handle both POST form data and JSON input
$input = json_decode(file_get_contents('php://input'), true);
logError('Datos recibidos', [
    'has_json_input' => !empty($input),
    'has_post_data' => !empty($_POST),
    'json_input' => $input,
    'post_data' => $_POST
]);

if ($input) {
    $lead_id = intval($input['lead_id'] ?? 0);
    $type = trim($input['type'] ?? 'note');
    $subject = trim($input['subject'] ?? '');
    $description = trim($input['description'] ?? '');
    $activity_date = trim($input['activity_date'] ?? '');
    $created_by = intval($input['created_by'] ?? $_SESSION['user_id']);
    logError('Usando datos JSON', compact('lead_id', 'type', 'subject', 'description', 'activity_date', 'created_by'));
} else {
    $lead_id = intval($_POST['lead_id'] ?? 0);
    $type = trim($_POST['type'] ?? 'note');
    $subject = trim($_POST['subject'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $activity_date = trim($_POST['activity_date'] ?? '');
    $created_by = intval($_POST['created_by'] ?? $_SESSION['user_id']);
    logError('Usando datos POST', compact('lead_id', 'type', 'subject', 'description', 'activity_date', 'created_by'));
}

if ($lead_id <= 0) {
    logError('lead_id inválido', ['lead_id' => $lead_id]);
    http_response_code(400);
    ob_end_clean(); // Limpiar buffer antes de enviar JSON
    echo json_encode(['success' => false, 'message' => 'lead_id inválido']);
    exit;
}

if ($subject === '') {
    logError('Asunto vacío', ['subject' => $subject]);
    http_response_code(400);
    ob_end_clean(); // Limpiar buffer antes de enviar JSON
    echo json_encode(['success' => false, 'message' => 'El asunto es obligatorio']);
    exit;
}

logError('Validaciones básicas pasadas', compact('lead_id', 'subject'));

// Validate activity date format if provided
if ($activity_date && !strtotime($activity_date)) {
    $activity_date = date('Y-m-d H:i:s');
} elseif (!$activity_date) {
    $activity_date = date('Y-m-d H:i:s');
}

// Validar existencia del lead
$lead = $db->selectOne('SELECT id FROM leads WHERE id = ?', [$lead_id]);
logError('Verificación de lead', ['lead_found' => !empty($lead), 'lead_data' => $lead]);

if (!$lead) {
    logError('Lead no encontrado', ['lead_id' => $lead_id]);
    http_response_code(404);
    ob_end_clean(); // Limpiar buffer antes de enviar JSON
    echo json_encode(['success' => false, 'message' => 'Lead no encontrado']);
    exit;
}

logError('Lead encontrado, procediendo con inserción de actividad');

try {
    logError('Iniciando inserción de actividad', [
        'query_params' => [$lead_id, $type, $subject, $description, $created_by, $activity_date]
    ]);
    
    $db->execute(
        "INSERT INTO lead_activities (lead_id, type, subject, description, status, created_by, created_at)
         VALUES (?, ?, ?, ?, 'completed', ?, ?)",
        [$lead_id, $type, $subject, $description, $created_by, $activity_date]
    );
    
    // Obtener el ID de la actividad insertada
    $activity_id = $db->lastInsertId();
    logError('Actividad insertada exitosamente', ['activity_id' => $activity_id]);
    
    // Actualizar solo el campo updated_at en leads (las otras columnas no existen)
    $db->execute(
        "UPDATE leads SET updated_at = ? WHERE id = ?",
        [$activity_date, $lead_id]
    );
    
    logError('Lead actualizado exitosamente');

    // Datos para respuesta
    $user = $db->selectOne('SELECT username, CONCAT(first_name, " ", last_name) AS name FROM users WHERE id = ?', [$created_by]);
    logError('Usuario obtenido para respuesta', ['user_data' => $user]);

    $response = [
        'success' => true,
        'message' => 'Comentario agregado exitosamente',
        'activity' => [
            'id' => $activity_id,
            'created_at' => $activity_date,
            'activity_date' => $activity_date,
            'type' => $type,
            'subject' => $subject,
            'description' => $description,
            'creator_username' => $user['username'] ?? '',
            'creator_name' => $user['name'] ?? ''
        ]
    ];
    
    logError('Respuesta exitosa preparada', ['response' => $response]);
    ob_end_clean(); // Limpiar buffer antes de enviar JSON
    echo json_encode($response);

} catch (Exception $e) {
    logError('Error en inserción de actividad', [
        'error_message' => $e->getMessage(),
        'error_code' => $e->getCode(),
        'error_file' => $e->getFile(),
        'error_line' => $e->getLine(),
        'stack_trace' => $e->getTraceAsString(),
        'query_params' => [$lead_id, $type, $subject, $description, $created_by, $activity_date]
    ]);
    
    error_log("Error en add_note.php: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    error_log("POST data: " . print_r($_POST, true));
    error_log("JSON input: " . file_get_contents('php://input'));
    
    http_response_code(500);
    ob_end_clean(); // Limpiar buffer antes de enviar JSON
    echo json_encode([
        'success' => false, 
        'message' => 'Error de conexión con el servidor: ' . $e->getMessage(),
        'error_details' => $e->getTraceAsString(),
        'debug_info' => [
            'post_data' => $_POST,
            'json_input' => file_get_contents('php://input'),
            'session_user_id' => $_SESSION['user_id'] ?? 'no_session',
            'error_file' => $e->getFile(),
            'error_line' => $e->getLine()
        ]
    ]);
}

} catch (Exception $mainException) {
    logError('Error principal en add_note.php', [
        'error_message' => $mainException->getMessage(),
        'error_code' => $mainException->getCode(),
        'error_file' => $mainException->getFile(),
        'error_line' => $mainException->getLine(),
        'stack_trace' => $mainException->getTraceAsString()
    ]);
    
    http_response_code(500);
    ob_end_clean(); // Limpiar buffer antes de enviar JSON
    echo json_encode([
        'success' => false, 
        'message' => 'Error de conexión con el servidor: ' . $mainException->getMessage(),
        'error_details' => $mainException->getTraceAsString()
    ]);
}

logError('Request finalizada');

// El buffer se limpia antes de cada respuesta JSON ahora