// Wizard Modal Variables
let currentStep = 1;
let selectedCommentType = '';
let commentData = {}

// Show result modal
function showResultModal(activity) {
    const modal = document.getElementById('resultModal');
    
    // Update modal content
    document.getElementById('resultSubject').textContent = activity.subject;
    document.getElementById('resultDescription').textContent = activity.description;
    document.getElementById('resultType').textContent = activity.type;
    document.getElementById('resultDate').textContent = activity.activity_date;
    
    // Show modal
    modal.classList.add('show');
}

// Close result modal
function closeResultModal() {
    const modal = document.getElementById('resultModal');
    modal.classList.remove('show');
}

// Event listeners for result modal
document.addEventListener('DOMContentLoaded', function() {
    const resultModal = document.getElementById('resultModal');
    const closeBtn = document.querySelector('.result-close-btn');
    const okBtn = document.querySelector('.result-btn-primary');
    
    if (closeBtn) {
        closeBtn.addEventListener('click', closeResultModal);
    }
    
    if (okBtn) {
        okBtn.addEventListener('click', closeResultModal);
    }
    
    // Close modal when clicking outside
    if (resultModal) {
        resultModal.addEventListener('click', function(e) {
            if (e.target === resultModal) {
                closeResultModal();
            }
        });
    }
});;

// Open Comment Wizard
function openCommentWizard() {
    const modal = document.getElementById('commentWizardModal');
    
    // Reset wizard state
    currentStep = 1;
    selectedCommentType = '';
    commentData = {};
    
    // Reset UI
    resetWizardSteps();
    updateProgressSteps();
    updateNavigationButtons();
    
    // Set current date/time
    const now = new Date();
    const localDateTime = now.toISOString().slice(0, 16);
    document.getElementById('commentDate').value = localDateTime;
    
    // Show modal with animation
    modal.style.display = 'flex';
    setTimeout(() => {
        modal.classList.add('show');
    }, 10);
}

// Close Comment Wizard
function closeCommentWizard() {
    const modal = document.getElementById('commentWizardModal');
    modal.classList.remove('show');
    
    setTimeout(() => {
        modal.style.display = 'none';
        resetWizardSteps();
    }, 300);
}

// Reset wizard to initial state
function resetWizardSteps() {
    // Reset steps
    document.querySelectorAll('.wizard-step').forEach(step => {
        step.classList.remove('active');
    });
    document.querySelector('.wizard-step[data-step="1"]').classList.add('active');
    
    // Reset progress
    document.querySelectorAll('.progress-step').forEach(step => {
        step.classList.remove('active', 'completed');
    });
    document.querySelector('.progress-step[data-step="1"]').classList.add('active');
    
    // Reset form
    document.getElementById('commentForm').reset();
    
    // Reset selected type
    document.querySelectorAll('.comment-type-card').forEach(card => {
        card.classList.remove('selected');
    });
    
    currentStep = 1;
    selectedCommentType = '';
}

// Update progress steps visual state
function updateProgressSteps() {
    document.querySelectorAll('.progress-step').forEach(step => {
        const stepNumber = parseInt(step.dataset.step);
        
        step.classList.remove('active', 'completed');
        
        if (stepNumber < currentStep) {
            step.classList.add('completed');
        } else if (stepNumber === currentStep) {
            step.classList.add('active');
        }
    });
}

// Update navigation buttons
function updateNavigationButtons() {
    const prevBtn = document.getElementById('prevBtn');
    const nextBtn = document.getElementById('nextBtn');
    const saveBtn = document.getElementById('saveBtn');
    
    // Previous button
    if (currentStep === 1) {
        prevBtn.style.display = 'none';
    } else {
        prevBtn.style.display = 'block';
    }
    
    // Next/Save buttons
    if (currentStep === 3) {
        nextBtn.style.display = 'none';
        saveBtn.style.display = 'block';
    } else {
        nextBtn.style.display = 'block';
        saveBtn.style.display = 'none';
    }
    
    // Enable/disable next button based on step validation
    nextBtn.disabled = !validateCurrentStep();
}

// Validate current step
function validateCurrentStep() {
    switch (currentStep) {
        case 1:
            return selectedCommentType !== '';
        case 2:
            const subject = document.getElementById('commentSubject').value.trim();
            const description = document.getElementById('commentDescription').value.trim();
            return subject !== '' && description !== '';
        case 3:
            return true;
        default:
            return false;
    }
}

// Next step
function nextStep() {
    if (!validateCurrentStep()) {
        return;
    }
    
    if (currentStep < 3) {
        // Save current step data
        saveStepData();
        
        // Move to next step
        currentStep++;
        showStep(currentStep);
        updateProgressSteps();
        updateNavigationButtons();
        
        // Update preview if on step 3
        if (currentStep === 3) {
            updatePreview();
        }
    }
}

// Previous step
function previousStep() {
    if (currentStep > 1) {
        currentStep--;
        showStep(currentStep);
        updateProgressSteps();
        updateNavigationButtons();
    }
}

// Show specific step
function showStep(stepNumber) {
    document.querySelectorAll('.wizard-step').forEach(step => {
        step.classList.remove('active');
    });
    
    document.querySelector(`.wizard-step[data-step="${stepNumber}"]`).classList.add('active');
}

// Save current step data
function saveStepData() {
    switch (currentStep) {
        case 1:
            commentData.type = selectedCommentType;
            break;
        case 2:
            commentData.subject = document.getElementById('commentSubject').value.trim();
            commentData.description = document.getElementById('commentDescription').value.trim();
            commentData.date = document.getElementById('commentDate').value;
            break;
    }
}

// Update preview in step 3
function updatePreview() {
    const typeNames = {
        'call': '📞 Llamada',
        'email': '📧 Email',
        'meeting': '🤝 Reunión',
        'note': '📝 Nota'
    };
    
    document.getElementById('previewType').textContent = typeNames[commentData.type] || '';
    document.getElementById('previewSubject').textContent = commentData.subject || '';
    document.getElementById('previewDescription').textContent = commentData.description || '';
    
    // Format date
    if (commentData.date) {
        const date = new Date(commentData.date);
        const formattedDate = date.toLocaleString('es-ES', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
            hour: '2-digit',
            minute: '2-digit'
        });
        document.getElementById('previewDate').textContent = formattedDate;
    }
}

// Save comment to database
async function saveComment() {
    const saveBtn = document.getElementById('saveBtn');
    
    // Show loading state
    saveBtn.classList.add('loading');
    saveBtn.disabled = true;
    
    try {
        // Validate required data
        if (!window.leadData || !window.leadData.id) {
            throw new Error('ID del lead no disponible');
        }
        
        if (!commentData.type || !commentData.subject || !commentData.description) {
            throw new Error('Faltan datos requeridos del comentario');
        }
        
        // Prepare data for API
        const formData = {
            lead_id: window.leadData.id,
            type: commentData.type,
            subject: commentData.subject,
            description: commentData.description,
            activity_date: commentData.date,
            created_by: window.currentUserId || 1
        };
        
        console.log('Enviando datos:', formData);
        
        // Send to server with correct module path
        const response = await fetch('modules/leads/add_note.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: JSON.stringify(formData)
        });
        
        console.log('Response status:', response.status);
        
        if (response.ok) {
            // Obtener el texto crudo primero para debugging
            const responseText = await response.text();
            console.log('Response text:', responseText);
            
            try {
                const result = JSON.parse(responseText);
                console.log('Response data:', result);
                
                if (result.success) {
                    // Show result modal instead of adding to timeline
                    showResultModal(result.activity);
                    
                    // Close wizard
                    closeCommentWizard();
                    
                    // Agregar la nueva actividad al timeline en tiempo real
                    if (result.activity) {
                        addActivityToTimeline(result.activity);
                    }
                    // Actualizar el contador de actividades si existe
                    updateActivityCount();
                } else {
                    throw new Error(result.message || 'Error al guardar el comentario');
                }
            } catch (jsonError) {
                console.error('Error al parsear JSON:', jsonError);
                console.error('Texto de respuesta:', responseText);
                throw new Error(`Error al procesar la respuesta del servidor: ${jsonError.message}. Respuesta cruda: ${responseText.substring(0, 200)}`);
            }
        } else {
            const errorText = await response.text();
            console.error('Server error response:', errorText);
            throw new Error(`Error del servidor (${response.status}): ${errorText}`);
        }
        
    } catch (error) {
        console.error('Error completo:', error);
        showNotification('❌ Error al agregar el comentario: ' + error.message, 'error');
    } finally {
        // Reset button state
        saveBtn.classList.remove('loading');
        saveBtn.disabled = false;
    }
}

// Add activity to timeline (enhanced version)
function addActivityToTimeline(activityData) {
    const timeline = document.getElementById('activityTimeline');
    
    if (!timeline) {
        console.warn('Timeline element not found');
        return;
    }
    
    // Create timeline header if it doesn't exist
    if (!timeline.querySelector('.timeline-header')) {
        const header = document.createElement('div');
        header.className = 'timeline-header';
        header.innerHTML = `
            <div class="timeline-column">Fecha de Creación</div>
            <div class="timeline-column">Actividad</div>
            <div class="timeline-column">Comentario</div>
            <div class="timeline-column">Creado Por</div>
        `;
        timeline.appendChild(header);
    }
    
    // Create new activity row
    const activityRow = document.createElement('div');
    activityRow.className = 'timeline-row new-activity';
    
    // Format date
    const activityDate = activityData.activity_date ? 
        new Date(activityData.activity_date).toLocaleString('es-ES', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
            hour: '2-digit',
            minute: '2-digit'
        }) : 
        new Date().toLocaleString('es-ES', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
            hour: '2-digit',
            minute: '2-digit'
        });
    
    // Activity type icons
    const typeIcons = {
        'call': '📞',
        'email': '📧',
        'meeting': '🤝',
        'note': '📝'
    };
    
    const typeNames = {
        'call': 'Llamada',
        'email': 'Email',
        'meeting': 'Reunión',
        'note': 'Nota'
    };
    
    activityRow.innerHTML = `
        <div class="timeline-column">
            <span class="activity-date">${activityDate}</span>
        </div>
        <div class="timeline-column">
            <div class="activity-type">
                <span class="activity-icon">${typeIcons[activityData.type] || '📝'}</span>
                <span class="activity-name">${typeNames[activityData.type] || 'Actividad'}</span>
            </div>
        </div>
        <div class="timeline-column">
            <div class="activity-description">${activityData.description}</div>
        </div>
        <div class="timeline-column">
            <span class="activity-user">${window.currentUserName || 'Usuario'}</span>
        </div>
    `;
    
    // Insert at the beginning (after header)
    const header = timeline.querySelector('.timeline-header');
    if (header && header.nextSibling) {
        timeline.insertBefore(activityRow, header.nextSibling);
    } else {
        timeline.appendChild(activityRow);
    }
    
    // Add animation
    setTimeout(() => {
        activityRow.classList.add('animate-in');
    }, 100);
    
    // Remove new-activity class after animation
    setTimeout(() => {
        activityRow.classList.remove('new-activity');
    }, 1000);
}

// Show notification
function showNotification(message, type = 'info') {
    // Remove existing notifications
    document.querySelectorAll('.notification').forEach(n => n.remove());
    
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    
    // Style the notification
    Object.assign(notification.style, {
        position: 'fixed',
        top: '20px',
        right: '20px',
        padding: '16px 24px',
        borderRadius: '8px',
        color: 'white',
        fontWeight: '600',
        zIndex: '10000',
        transform: 'translateX(100%)',
        transition: 'transform 0.3s ease',
        maxWidth: '400px',
        boxShadow: '0 4px 12px rgba(0, 0, 0, 0.2)'
    });
    
    // Set background color based on type
    const colors = {
        'success': '#28a745',
        'error': '#dc3545',
        'info': '#007bff',
        'warning': '#ffc107'
    };
    
    notification.style.background = colors[type] || colors.info;
    
    document.body.appendChild(notification);
    
    // Animate in
    setTimeout(() => {
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            notification.remove();
        }, 300);
    }, 5000);
}

// Event Listeners
document.addEventListener('DOMContentLoaded', function() {
    // Comment type selection
    document.querySelectorAll('.comment-type-card').forEach(card => {
        card.addEventListener('click', function() {
            // Remove previous selection
            document.querySelectorAll('.comment-type-card').forEach(c => {
                c.classList.remove('selected');
            });
            
            // Select current card
            this.classList.add('selected');
            selectedCommentType = this.dataset.type;
            
            // Update navigation
            updateNavigationButtons();
        });
    });
    
    // Form input validation
    document.getElementById('commentSubject').addEventListener('input', updateNavigationButtons);
    document.getElementById('commentDescription').addEventListener('input', updateNavigationButtons);
    
    // Prevent form submission
    document.getElementById('commentForm').addEventListener('submit', function(e) {
        e.preventDefault();
    });
    
    // Close modal on outside click
    document.getElementById('commentWizardModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeCommentWizard();
        }
    });
    
    // Escape key to close modal
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            const modal = document.getElementById('commentWizardModal');
            if (modal.classList.contains('show')) {
                closeCommentWizard();
            }
        }
    });
});

// Update quick action cards to use new wizard
document.addEventListener('DOMContentLoaded', function() {
    // Update "Agregar Notas" card
    const addNotesCard = document.querySelector('[onclick*="openNoteModal"]');
    if (addNotesCard) {
        addNotesCard.setAttribute('onclick', 'openCommentWizard()');
    }
    
    // Update other activity cards if they exist
    const activityCards = document.querySelectorAll('[onclick*="openActivityModal"]');
    activityCards.forEach(card => {
        card.setAttribute('onclick', 'openCommentWizard()');
    });
});

// Global variables for lead data
// Note: These should be set by PHP in the main view file
window.currentUserId = window.currentUserId || 1;
window.currentUserName = window.currentUserName || 'Usuario';

// Función para cargar actividades con AJAX
function loadActivitiesPage(page) {
    const leadId = window.leadData?.id;
    if (!leadId) return;
    
    // Mostrar indicador de carga
    const timeline = document.getElementById('activityTimeline');
    if (timeline) {
        timeline.innerHTML = '<div class="loading-activities">Cargando actividades...</div>';
    }
    
    fetch(`modules/leads/load_activities_ajax.php?lead_id=${leadId}&page=${page}`, {
        method: 'GET',
        credentials: 'same-origin', // Incluir cookies de sesión
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Actualizar el contenido del timeline
                if (timeline) {
                    timeline.innerHTML = data.html;
                }
                
                // Actualizar la paginación
                const paginationContainer = document.querySelector('.pagination-controls');
                if (paginationContainer && data.pagination_html) {
                    paginationContainer.innerHTML = data.pagination_html;
                }
                
                // Actualizar el contador si existe
                updateActivityCount(data.total_activities);
            } else {
                showNotification(data.message || 'Error al cargar actividades', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Error al cargar actividades', 'error');
        });
}

// Función para actualizar el contador de actividades
function updateActivityCount(count) {
    // Buscar elementos que puedan mostrar el contador de actividades
    const statCards = document.querySelectorAll('.stat-card');
    statCards.forEach(card => {
        const label = card.querySelector('.stat-label');
        if (label && (label.textContent.includes('Actividades') || label.textContent.includes('Actividad'))) {
            const number = card.querySelector('.stat-number');
            if (number) {
                number.textContent = count || '0';
            }
        }
    });
}

// Agregar estilos CSS para la paginación y carga
const ajaxStyles = document.createElement('style');
ajaxStyles.textContent = `
    .loading-activities {
        text-align: center;
        padding: 40px;
        color: #666;
        font-size: 16px;
    }
    
    .pagination-container {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 8px;
        margin-top: 20px;
        flex-wrap: wrap;
    }
    
    .pagination-btn {
        padding: 8px 12px;
        border: 1px solid #ddd;
        background: white;
        color: #333;
        border-radius: 6px;
        cursor: pointer;
        font-size: 14px;
        transition: all 0.2s ease;
        min-width: 40px;
    }
    
    .pagination-btn:hover {
        background-color: #f8f9fa;
        border-color: #007bff;
        color: #007bff;
    }
    
    .pagination-btn.pagination-active {
        background-color: #007bff;
        color: white;
        border-color: #007bff;
    }
    
    .pagination-btn.pagination-active:hover {
        background-color: #0056b3;
        border-color: #0056b3;
    }
    
    .pagination-btn:disabled {
        opacity: 0.5;
        cursor: not-allowed;
    }
    
    .pagination-dots {
        padding: 8px 4px;
        color: #666;
        font-size: 14px;
    }
    
    .pagination-prev, .pagination-next {
        font-weight: 500;
        padding: 8px 16px;
    }
    
    @keyframes fadeIn {
        from { opacity: 0; transform: translateY(-10px); }
        to { opacity: 1; transform: translateY(0); }
    }
    
    .new-activity {
        animation: fadeIn 0.5s ease-in;
    }
`;
document.head.appendChild(ajaxStyles);