/**
 * ProfixCRM Simple - JavaScript Principal
 * Funcionalidades generales del sistema
 */

// Inicialización del sistema
document.addEventListener('DOMContentLoaded', function() {
    initializeSystem();
    initializeTables();
    initializeAuthUX();
});

function initializeSystem() {
    // Inicializar componentes básicos
    initializeSidebar();
    initializeTheme();
    initializeModals();
    
    console.log('ProfixCRM Simple inicializado correctamente');
}

// ===== Tablas y listas (UX) =====
function initializeTables() {
    const tables = Array.from(document.querySelectorAll('table.data-table, table.leads-table'));
    tables.forEach(setupTableUX);
}

function setupTableUX(table) {
    if (!table) return;

    // Hover y clic de fila (navega al primer enlace "Ver")
    Array.from(table.tBodies[0]?.rows || []).forEach(row => {
        row.classList.add('row-hover');
        row.addEventListener('click', (e) => {
            // Evitar interferir con enlaces/botones y elementos con clase lead-name-link
            if (e.target.closest('a, button, input, select, label, .lead-name-link')) return;
            
            // Prevenir si el clic fue en una celda que contiene enlaces de nombres
            if (e.target.closest('.lead-first-name, .lead-last-name')) return;
            
            const viewLink = row.querySelector('a[href*="action=view"], .btn[href*="action=view"]');
            if (viewLink && viewLink.href) {
                window.location.href = viewLink.href;
            }
        });
    });

    // Ordenación básica por columna
    const thead = table.tHead;
    if (thead) {
        Array.from(thead.rows[0]?.cells || []).forEach((th, idx) => {
            th.classList.add('sortable');
            if (!th.hasAttribute('onclick')) {
                th.addEventListener('click', () => sortTableByColumn(table, idx));
            }
        });
    }
}

function sortTableByColumn(table, columnIndex) {
    const tbody = table.tBodies[0];
    if (!tbody) return;
    const rows = Array.from(tbody.rows);
    const asc = table.getAttribute('data-sort') !== `col-${columnIndex}-asc`;
    rows.sort((a, b) => {
        const av = (a.cells[columnIndex]?.textContent || '').trim().toLowerCase();
        const bv = (b.cells[columnIndex]?.textContent || '').trim().toLowerCase();
        // Si es número, comparar como número
        const an = parseFloat(av.replace(/[^0-9.-]/g, ''));
        const bn = parseFloat(bv.replace(/[^0-9.-]/g, ''));
        if (!isNaN(an) && !isNaN(bn)) return asc ? an - bn : bn - an;
        return asc ? av.localeCompare(bv) : bv.localeCompare(av);
    });
    rows.forEach(r => tbody.appendChild(r));
    table.setAttribute('data-sort', asc ? `col-${columnIndex}-asc` : `col-${columnIndex}-desc`);
}

// Navegación a perfil de lead
function goToLeadProfile(id) {
    if (!id) return;
    window.location.href = `/index.php?module=leads&action=view&id=${encodeURIComponent(id)}`;
}

function viewLead(id) { // compatibilidad con otras vistas
    goToLeadProfile(id);
}

// Gestión del sidebar
function initializeSidebar() {
    const sidebarToggle = document.querySelector('.sidebar-toggle');
    const sidebar = document.querySelector('.sidebar');
    
    if (sidebarToggle && sidebar) {
        sidebarToggle.addEventListener('click', function() {
            sidebar.classList.toggle('collapsed');
        });
    }
}

// Gestión del tema
function initializeTheme() {
    const themeToggle = document.querySelector('.theme-toggle');
    
    if (themeToggle) {
        themeToggle.addEventListener('click', function() {
            document.body.classList.toggle('dark-theme');
            
            // Guardar preferencia
            const isDark = document.body.classList.contains('dark-theme');
            localStorage.setItem('theme', isDark ? 'dark' : 'light');
        });
        
        // Cargar tema guardado
        const savedTheme = localStorage.getItem('theme');
        if (savedTheme === 'dark') {
            document.body.classList.add('dark-theme');
        }
    }
}

// Gestión de modales
function initializeModals() {
    // Cerrar modales con ESC
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            const openModal = document.querySelector('.modal:not([style*="display: none"])');
            if (openModal) {
                closeModal(openModal);
            }
        }
    });
    
    // Cerrar modales clickeando fuera
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('modal')) {
            closeModal(e.target);
        }
    });
}

function closeModal(modal) {
    if (modal && modal.parentNode) {
        modal.parentNode.removeChild(modal);
    }
}

// Utilidades generales
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    // Auto-remover después de 3 segundos
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 3000);
}

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('es-ES', {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Funciones específicas para módulos
function refreshTable(tableId) {
    const table = document.getElementById(tableId);
    if (table) {
        // Lógica para refrescar tabla
        console.log(`Refrescando tabla: ${tableId}`);
    }
}

function confirmAction(message, callback) {
    // Modal de confirmación en lugar de confirm() nativo
    showModalNotification({
        type: 'info',
        title: 'Confirmación',
        message,
        actions: [
            { text: 'Cancelar', variant: 'secondary' },
            { text: 'Confirmar', variant: 'primary', onClick: () => callback() }
        ]
    });
}

// Exportar funciones globales
window.ProfixCRM = {
    showNotification,
    showModalNotification,
    formatDate,
    escapeHtml,
    refreshTable,
    confirmAction,
    closeModal,
    goToLeadProfile,
    viewLead
};

// Notificaciones Modales
function showModalNotification({ type = 'info', title = '', message = '', actions = [] } = {}) {
    // Backdrop
    const backdrop = document.createElement('div');
    backdrop.className = 'modal-backdrop';
    // Modal
    const modal = document.createElement('div');
    modal.className = 'modal modal-premium modal-notification modal-' + type;
    modal.innerHTML = `
        <div class="modal-header">
            <div class="modal-title">
                <span class="modal-icon">${getIconForType(type)}</span>
                <h3>${escapeHtml(title || capitalize(type))}</h3>
            </div>
            <button class="btn btn-ghost" data-close>&times;</button>
        </div>
        <div class="modal-body">
            <p>${escapeHtml(message)}</p>
        </div>
        <div class="modal-footer"></div>
    `;
    // Footer actions
    const footer = modal.querySelector('.modal-footer');
    if (Array.isArray(actions) && actions.length) {
        actions.forEach((a) => {
            const btn = document.createElement('button');
            btn.className = 'btn btn-' + (a.variant || 'secondary');
            btn.textContent = a.text || 'OK';
            btn.addEventListener('click', () => {
                try { a.onClick && a.onClick(); } finally { document.body.removeChild(backdrop); }
            });
            footer.appendChild(btn);
        });
    } else {
        const btn = document.createElement('button');
        btn.className = 'btn btn-primary';
        btn.textContent = 'Aceptar';
        btn.addEventListener('click', () => document.body.removeChild(backdrop));
        footer.appendChild(btn);
    }
    // Cerrar
    modal.querySelector('[data-close]').addEventListener('click', () => document.body.removeChild(backdrop));
    backdrop.addEventListener('click', (e) => { if (e.target === backdrop) document.body.removeChild(backdrop); });
    backdrop.appendChild(modal);
    document.body.appendChild(backdrop);
}

// ===== Auth UX (Login) =====
function initializeAuthUX() {
    // Toggle mostrar/ocultar contraseña
    const pwdInput = document.getElementById('password');
    const toggle = document.getElementById('togglePassword');
    if (pwdInput && toggle) {
        toggle.addEventListener('click', () => {
            const isText = pwdInput.type === 'text';
            pwdInput.type = isText ? 'password' : 'text';
            toggle.textContent = isText ? 'Mostrar' : 'Ocultar';
        });
    }
}

function getIconForType(type) {
    switch (type) {
        case 'success': return '✅';
        case 'error': return '⛔';
        case 'warning': return '⚠️';
        default: return 'ℹ️';
    }
}

function capitalize(s){ return (s||'').charAt(0).toUpperCase() + (s||'').slice(1); }