<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🐛 Debug - Editar Rol</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
        h1 {
            text-align: center;
            color: #2c3e50;
            margin-bottom: 30px;
        }
        .test-section {
            margin: 20px 0;
            padding: 20px;
            border: 2px solid #3498db;
            border-radius: 10px;
            background: #f8f9fa;
        }
        .test-button {
            background: #3498db;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            margin: 5px;
            font-size: 14px;
        }
        .test-button:hover {
            background: #2980b9;
        }
        .result {
            margin: 10px 0;
            padding: 10px;
            border-radius: 5px;
            font-family: monospace;
            font-size: 12px;
        }
        .success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        .warning {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🐛 Debug - Error al Editar Rol</h1>
        
        <div class="test-section">
            <h3>📋 Información del Problema</h3>
            <p><strong>Error reportado:</strong> "Unexpected token '<', "<!DOCTYPE "... is not valid JSON"</p>
            <p><strong>Acción:</strong> Al hacer clic en "Editar" en el módulo de roles</p>
            <p><strong>Endpoint:</strong> <code>actions.php?action=get_role_details&role_id=X</code></p>
        </div>

        <div class="test-section">
            <h3>🔍 Tests de Diagnóstico</h3>
            
            <button class="test-button" onclick="testSessionStatus()">1. Verificar Estado de Sesión</button>
            <button class="test-button" onclick="testDatabaseConnection()">2. Probar Conexión BD</button>
            <button class="test-button" onclick="testEndpointDirect()">3. Test Endpoint Directo</button>
            <button class="test-button" onclick="testWithSession()">4. Test con Sesión Simulada</button>
            <button class="test-button" onclick="testEditRoleFunction()">5. Simular Función editRole()</button>
            
            <div id="testResults"></div>
        </div>

        <div class="test-section">
            <h3>🛠️ Soluciones Aplicadas</h3>
            <ul>
                <li>✅ Rutas AJAX corregidas (absoluta → relativa)</li>
                <li>✅ Configuración local de BD creada</li>
                <li>⚠️ Problema de autenticación detectado</li>
            </ul>
        </div>
    </div>

    <script>
        function addResult(message, type = 'info') {
            const container = document.getElementById('testResults');
            const result = document.createElement('div');
            result.className = `result ${type}`;
            result.innerHTML = `[${new Date().toLocaleTimeString()}] ${message}`;
            container.appendChild(result);
        }

        async function testSessionStatus() {
            addResult('🔄 Verificando estado de sesión...', 'info');
            
            try {
                const response = await fetch('../../debug_session_quick.php');
                const text = await response.text();
                
                if (text.includes('user_id')) {
                    addResult('✅ Sesión activa detectada', 'success');
                } else {
                    addResult('❌ No hay sesión activa - Este es el problema principal', 'error');
                    addResult('💡 Solución: Inicia sesión en el sistema antes de usar roles', 'warning');
                }
            } catch (error) {
                addResult(`❌ Error al verificar sesión: ${error.message}`, 'error');
            }
        }

        async function testDatabaseConnection() {
            addResult('🔄 Probando conexión a base de datos...', 'info');
            
            try {
                const response = await fetch('../../database/connection.php');
                const text = await response.text();
                
                if (response.ok) {
                    addResult('✅ Archivo de conexión accesible', 'success');
                } else {
                    addResult('❌ Problema con archivo de conexión', 'error');
                }
            } catch (error) {
                addResult(`❌ Error de conexión BD: ${error.message}`, 'error');
            }
        }

        async function testEndpointDirect() {
            addResult('🔄 Probando endpoint get_role_details directamente...', 'info');
            
            try {
                const response = await fetch('modules/roles/actions.php?action=get_role_details&role_id=1');
                const text = await response.text();
                
                addResult(`📄 Respuesta del servidor: ${text.substring(0, 200)}...`, 'info');
                
                if (text.startsWith('<!DOCTYPE')) {
                    addResult('❌ El servidor devuelve HTML en lugar de JSON', 'error');
                    addResult('🔍 Esto indica un error PHP o redirección', 'warning');
                } else if (text.includes('No autorizado')) {
                    addResult('❌ Error de autorización - Falta sesión', 'error');
                } else {
                    try {
                        JSON.parse(text);
                        addResult('✅ Respuesta JSON válida', 'success');
                    } catch {
                        addResult('❌ Respuesta no es JSON válido', 'error');
                    }
                }
            } catch (error) {
                addResult(`❌ Error en endpoint: ${error.message}`, 'error');
            }
        }

        async function testWithSession() {
            addResult('🔄 Intentando crear sesión temporal...', 'info');
            
            // Primero intentar login
            try {
                const loginData = new FormData();
                loginData.append('username', 'admin');
                loginData.append('password', 'admin');
                
                const loginResponse = await fetch('modules/auth/login.php', {
                    method: 'POST',
                    body: loginData
                });
                
                const loginText = await loginResponse.text();
                addResult(`📄 Respuesta login: ${loginText.substring(0, 100)}...`, 'info');
                
                // Ahora probar el endpoint de roles
                const roleResponse = await fetch('modules/roles/actions.php?action=get_role_details&role_id=1');
                const roleText = await roleResponse.text();
                
                if (roleText.includes('No autorizado')) {
                    addResult('❌ Aún sin autorización después del login', 'error');
                } else {
                    addResult('✅ Autorización exitosa', 'success');
                }
                
            } catch (error) {
                addResult(`❌ Error en test con sesión: ${error.message}`, 'error');
            }
        }

        async function testEditRoleFunction() {
            addResult('🔄 Simulando función editRole() del frontend...', 'info');
            
            // Simular exactamente lo que hace la función editRole
            try {
                const roleId = 1;
                const resp = await fetch(`modules/roles/actions.php?action=get_role_details&role_id=${encodeURIComponent(roleId)}`);
                
                addResult(`📊 Status: ${resp.status} ${resp.statusText}`, 'info');
                addResult(`📋 Headers: ${resp.headers.get('content-type')}`, 'info');
                
                const text = await resp.text();
                addResult(`📄 Respuesta completa: ${text}`, 'info');
                
                // Intentar parsear como JSON
                try {
                    const data = JSON.parse(text);
                    addResult('✅ JSON parseado exitosamente', 'success');
                    addResult(`📊 Datos: ${JSON.stringify(data, null, 2)}`, 'success');
                } catch (parseError) {
                    addResult(`❌ Error al parsear JSON: ${parseError.message}`, 'error');
                    
                    if (text.startsWith('<!DOCTYPE')) {
                        addResult('🔍 El servidor está devolviendo una página HTML de error', 'warning');
                        addResult('💡 Posibles causas: Error PHP, redirección, archivo no encontrado', 'warning');
                    }
                }
                
            } catch (networkError) {
                addResult(`❌ Error de red: ${networkError.message}`, 'error');
            }
        }

        // Auto-ejecutar algunos tests al cargar
        window.addEventListener('load', () => {
            setTimeout(() => {
                addResult('🚀 Iniciando diagnóstico automático...', 'info');
                testSessionStatus();
            }, 1000);
        });
    </script>
</body>
</html>